document.addEventListener("DOMContentLoaded", function () {
    const messages = Array.from(document.querySelectorAll(".message"));
    let currentMessageIndex = -1; // -1 = no message shown

    const THRESHOLD = 60; // pixels to trigger change
    let accumulated = 0;
    let timeoutId = null;
    let isScrolling = false; // debounce flag

    function updateMessageDisplay() {
        messages.forEach((m, i) => m.classList.toggle("active", i === currentMessageIndex));
    }

    function clampIndex(i) {
        return Math.max(-1, Math.min(messages.length - 1, i));
    }

    function onWheel(e) {
        // prevent page scroll while controlling messages
        e.preventDefault();

        // Skip if already scrolling (debounce)
        if (isScrolling) return;

        accumulated += e.deltaY;

        clearTimeout(timeoutId);

        if (accumulated >= THRESHOLD) {
            // scroll down -> next (one message only)
            isScrolling = true;
            currentMessageIndex = clampIndex(currentMessageIndex + 1);
            accumulated = 0;
            updateMessageDisplay();
        } else if (accumulated <= -THRESHOLD) {
            // scroll up -> previous (one message only)
            isScrolling = true;
            currentMessageIndex = clampIndex(currentMessageIndex - 1);
            accumulated = 0;
            updateMessageDisplay();
        }

        // Reset debounce after transition completes
        timeoutId = setTimeout(() => {
            accumulated = 0;
            isScrolling = false;
        }, 500);
    }

    // Support arrow keys for testing / accessibility
    function onKey(e) {
        if (e.key === "ArrowDown") {
            currentMessageIndex = clampIndex(currentMessageIndex + 1);
            updateMessageDisplay();
        } else if (e.key === "ArrowUp") {
            currentMessageIndex = clampIndex(currentMessageIndex - 1);
            updateMessageDisplay();
        }
    }

    // Mobile touch/swipe support
    let touchStartY = 0;
    let touchEndY = 0;

    function onTouchStart(e) {
        touchStartY = e.changedTouches[0].screenY;
    }

    function onTouchEnd(e) {
        touchEndY = e.changedTouches[0].screenY;
        handleSwipe();
    }

    function handleSwipe() {
        if (isScrolling) return;

        const diff = touchStartY - touchEndY; // negative = swipe up, positive = swipe down

        if (Math.abs(diff) > THRESHOLD) {
            isScrolling = true;
            if (diff > 0) {
                // swipe up -> next message
                currentMessageIndex = clampIndex(currentMessageIndex + 1);
            } else {
                // swipe down -> previous message
                currentMessageIndex = clampIndex(currentMessageIndex - 1);
            }
            updateMessageDisplay();

            timeoutId = setTimeout(() => {
                isScrolling = false;
            }, 500);
        }
    }

    window.addEventListener("wheel", onWheel, { passive: false });
    window.addEventListener("keydown", onKey);
    window.addEventListener("touchstart", onTouchStart, { passive: true });
    window.addEventListener("touchend", onTouchEnd, { passive: true });

    // initial state: no message visible (rose only)
    updateMessageDisplay();
});
